/* SCE CONFIDENTIAL
* PlayStation(R)3 Programmer Tool Runtime Library 430.001
* Copyright (C) 2012 Sony Computer Entertainment Inc.
* All Rights Reserved.
*/

#ifndef _SCE_FRIEND_MESSAGE_H
#define _SCE_FRIEND_MESSAGE_H

#include <list>
#include <string>
#include <np.h>
#include "np_toolkit_defines.h"

using namespace sce::Toolkit::NP::Utilities;

namespace sce {
	namespace Toolkit{
		namespace NP{

			/// @brief
			/// The namespace containing messaging functionality.
			///
			/// The namespace containing messaging functionality.
			namespace Messaging {

			/// @brief
			/// The static interface for sending messages.
			///
			/// The static interface for sending messages, which forwards commands on to the %NP %Toolkit thread.
			/// The %NP %Toolkit then attempts to fulfill the requests, and success or failure
			/// is notified by an event callback.
			///
			/// These functions have a corresponding set of messages that are passed to event callbacks to
			/// indicate asynchronous errors or events.
			///
			/// Events:
			/// <table>
			/// <tr><td>MESSAGE_ATTACHMENT_RECEIVED</td>		<td>A message with a data attachment was received.</td></tr>
			/// <tr><td>MESSAGE_INVITE_RECEIVED</td>			<td>A message with an invite was received.</td></tr>
			/// <tr><td>MESSAGE_XMB_INVITE_ACCEPTED</td>		<td>An invite was accepted via the XMB.</td></tr>
			/// </table>
			/// Results:
			/// <table>
			/// <tr><td>MESSAGE_ATTACHMENT_OPENED</td>			<td>A message with a data attachment was opened, and the sysutil GUI is now closed.</td></tr>
			/// <tr><td>MESSAGE_INVITE_ACCEPTED</td>			<td>An invite was accepted via the %NP %Toolkit, and the sysutil GUI is now closed.</td></tr>
			/// <tr><td>MESSAGE_SENT</td>						<td>A message was sent successfully.</td></tr>
			/// </table>
			///
			/// These Event messages can contain error codes.
		class Interface {
		public:
			/// @brief
			/// Sends a message to a remote user.
			///
			/// This function will cause the %NP %Toolkit thread to send a message to a remote user on the PSN.
			/// The function supports both custom data attachment messages and cross-title invites which are the 
			/// primary uses of the messaging system.
			/// Using the GUI is optional so long as all the arguments for the message are present.
			/// If anything is missing in the parameters and the GUI is enabled, the user will be prompted via System
			/// Software to enter the data manually.
			///
			/// @retval SCE_TOOLKIT_NP_SUCCESS					The operation completed successfully.
			/// @retval SCE_TOOLKIT_NP_INVALID_POINTER			The operation failed because one of the arguments had an incorrect value. For example, a string could have been too long.
			/// @retval SCE_TOOLKIT_NP_INVALID_POINTER			The operation failed because of <c>NULL</c> pointer exceptions.
			/// @retval SCE_TOOLKIT_NP_FAILED_ALLOCATE			The operation failed because memory could not be allocated on the heap.
			///
			/// @param msg						The message details including recipients, subject and body text.
			/// @param messageType				Specifies whether this is an invite or data message.
			/// @param gui						A flag that specifies whether to use the System GUI or not. Defaults to true.
			///
			/// @note
			/// This method is asynchronous.
			static int32_t sendMessage(const MessageData *msg,SceToolkitNpMessageType messageType,  bool gui = true );
	
			/// @brief
			/// Retrieves the attached data of an SceAppUtilAppEventParam message.
			///
			/// Retrieves the attached data of an SceAppUtilAppEventParam message.
			/// This function should be called when the an SceAppUtilAppEventParam of message is triggered.  
			/// When the operation has completed an event callback will be generated, and the <c>Future</c> object
			/// passed to <c><i>attch</i></c> needs to be polled to obtain the attached data.
			///
			/// @param attch		A <c>Future</c> object, which receives the attached data.
			/// @param invite		A flag that specifies whether to show invites or data messages to the user.
			///
			///	@retval	SCE_TOOLKIT_NP_SUCCESS				The operation was kicked off successfully.
			///	@retval	SCE_TOOLKIT_NP_INVALID_POINTER		The operation failed because an invalid pointer was passed to <c><i>attch</i></c>.
			/// @retval messageRetrieved					This event is passed to an event callback to signify the operation has completed successfully.
			/// @retval messageError						This event is passed to an event callback to signify that an error occurred during the operation.
			///
			/// @note
			/// This method is asynchronous.	
			static int32_t retrieveMessageAttachment(Future<MessageAttachment> *attch, bool invite);

			/// @brief
			/// Retrieves the data from an already accepted cross-title invite.
			///
			/// During execution of a title a user may receive a cross-title invite and open it
			/// from the XMB. The data is then returned to the application on the System Utility Callback.
			/// %NP %Toolkit will pick up this data and save it. The %NP %Toolkit Callback is then notified with
			/// the message MESSAGE_XMB_INVITE_ACCEPTED. Once this message is received this method is called
			/// to retrieve the cached invite data which the user has accepted via the XMB.
			/// This method is synchronous/blocking.
			///
			/// @param attch The buffer to copy the data to.
			///
			/// @retval SCE_TOOLKIT_NP_SUCCESS			The operation completed successfully.
			/// @retval SCE_TOOLKIT_NP_INVALID_POINTER	The operation failed because an invalid pointer was passed to <c><i>attch</i></c>.
			/// @retval	SCE_TOOLKIT_NP_NOT_INITIALISED	The operation failed because the %NP %Toolkit library is not initialized.
			static int32_t getAcceptedInvite(MessageAttachment *attch);

		}; //class 

			}//end messaging
		}//end NP
	}//end Toolkit 
}//end Sce
#endif
